#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <alloca.h>
#include <inttypes.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include <libdevmapper.h>

#include "blockdev.h"
#include "dmconvert.h"

#define BUFFER_SIZE	(1 << 20)
#define BOUNCE_SIZE	(1 << 22)

int conversion_setup(struct context *ctx, const char *source, const char *dest)
{
	const char *dmdir = dm_dir();
	char buf[DM_NAME_MAX + strlen(dmdir) + 2];
	unsigned int val;
	uint64_t val64;
	int src, dst;
	int pagesize = getpagesize();

	sprintf(buf, "%s/%s", dmdir, source);
	src = open(buf, O_RDONLY | O_DIRECT);
	if (src < 0) {
		perror(_("Could not open source device"));
		return -errno;
	}

	sprintf(buf, "%s/%s", dmdir, dest);
	dst = open(buf, O_WRONLY | O_DIRECT);
	if (dst < 0) {
		perror(_("Could not open destination device"));
		close(src);
		return -errno;
	}

	if (ioctl(src, BLKGETSIZE64, &val64) < 0) {
		perror(_("BLKGETSIZE64 ioctl failed on source device"));
		return -errno;
	}
	if (ctx->size == 0)
		ctx->size = val64;
	else if (ctx->size != val64) {
		fprintf(stderr, "Source device size does not match.\n");
		return -EINVAL;
	}

	if (ioctl(dst, BLKGETSIZE64, &val64) < 0) {
		perror(_("BLKGETSIZE64 ioctl failed on destination device"));
		return -errno;
	}
	if (ctx->size != val64) {
		fprintf(stderr, "Destination device size does not match.\n");
		return -EINVAL;
	}

	if (ioctl(src, BLKBSZGET, &val) < 0) {
		perror(_("BLKSZGET ioctl failed on source device"));
		return -errno;
	}
	ctx->bsize = val;

	if (ioctl(dst, BLKBSZGET, &val) < 0) {
		perror(_("BLKSZGET ioctl failed on destination device"));
		return -errno;
	}
	if (val > ctx->bsize)
		ctx->bsize = val;

	if (ctx->bsize < BUFFER_SIZE)
		ctx->bsize = BUFFER_SIZE;

	ctx->blocks = BOUNCE_SIZE / ctx->bsize;
	if (!ctx->blocks)
		ctx->blocks = 1;

	ctx->fd_source = src;
	ctx->fd_dest = dst;

	ctx->buffer = mmap(NULL, (ctx->bsize + pagesize - 1) & ~(pagesize - 1),
	                   PROT_READ | PROT_WRITE, MAP_PRIVATE | MAP_ANONYMOUS,
	                   0, 0);
	if (!ctx->buffer)
		return -ENOMEM;

	return 0;
}

int conversion_reopen_source(struct context *ctx)
{
	const char *dmdir = dm_dir();
	char buf[DM_NAME_MAX + strlen(dmdir) + 2];

	close(ctx->fd_source);

	sprintf(buf, "%s/%s", dmdir, DM_DEV_NAME(ctx->temps[DMC_DEV_SOURCE]));
	ctx->fd_source = open(buf, O_RDONLY | O_DIRECT);
	if (ctx->fd_source < 0) {
		/* hmm */
		mknod(buf, S_IFBLK | S_IRUSR | S_IWUSR | S_IRGRP,
		      MKDEV(ctx->temps[DMC_DEV_SOURCE].major,
		            ctx->temps[DMC_DEV_SOURCE].minor));

		ctx->fd_source = open(buf, O_RDONLY | O_DIRECT);
		if (ctx->fd_source < 0) {
			unlink(buf);
			perror(_("Could not open source device"));
			return -errno;
		}
		unlink(buf);
	}

	return 0;
}

int conversion_init(struct context *ctx, const char *source,
                    const char *dest, int resume)
{
	ctx->source = strdup(source);
	ctx->dest = dest ? strdup(dest) : NULL;

	atexit(dm_lib_release);

	if (resume) {
		struct dm_dev new_temps[DMC_DEV_COUNT];
		setup_temp_names(new_temps);
		if (capture_temp_targets(ctx, new_temps) < 0)
			return -EINVAL;
	} else {
		setup_temp_names(ctx->temps);
		if (swap_source_table(ctx, 0) < 0)
			return -EINVAL;
		if (rename_dest_target(ctx) < 0)
			return -EINVAL;
		if (setup_bounce_target(ctx) < 0)
			return -EINVAL;
	}

	return start_bounce_target(ctx);
}

static int do_transfer(struct context *ctx, unsigned int size)
{
	if (lseek(ctx->fd_source, ctx->offset, SEEK_SET) != ctx->offset)
		return -EIO;
	if (lseek(ctx->fd_dest, ctx->offset, SEEK_SET) != ctx->offset)
		return -EIO;
	if (read(ctx->fd_source, ctx->buffer, size) != size)
		return -EIO;
	if (write(ctx->fd_dest, ctx->buffer, size) != size)
		return -EIO;

	return 0;
}

int conversion_step(struct context *ctx)
{
	unsigned int i;
	uint64_t done = 0;

	if (step_bounce_target(ctx) < 0)
		return -EINVAL;
	if (init_device_bounce(ctx) < 0)
		return -EINVAL;

	for(i = 0; i < ctx->blocks; i++) {
		uint64_t size = ctx->size - ctx->offset;
		if (size > ctx->bsize)
			size = ctx->bsize;
		if (!size)
			break;

		if (do_transfer(ctx, (unsigned int)size) < 0)
			break;

		ctx->offset += size;
		done += size;
		if (ctx->offset >= ctx->size)
			break;
	}

	if (flush_bounce_target(ctx, done) < 0)
		return -EINVAL;
	if (finish_device_bounce(ctx) < 0)
		return -EINVAL;

	if (ctx->offset >= ctx->size)
		return 1;
	if (i < ctx->blocks)
		return -EINVAL;

	return 0;
}

int conversion_exit(struct context *ctx)
{
	int pagesize = getpagesize();

	close(ctx->fd_source);
	close(ctx->fd_dest);

	munmap(ctx->buffer, (ctx->bsize + pagesize - 1) & ~(pagesize - 1));

	if (ctx->offset >= ctx->size) {
		if (swap_source_table(ctx, 1) < 0)
			return -EINVAL;
		if (remove_temp_targets(ctx) < 0)
			return -EINVAL;
	} else {
		if (remove_bounce_target(ctx) < 0)
			return -EINVAL;
	}

	return 0;
}
