#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <sys/mman.h>
#include <errno.h>

#include <popt.h>

#include "dmconvert.h"

static int opt_force = 0;
static int opt_resume = 0;

static void usage(poptContext popt_context, int exitcode, const char *error, const char *more)
{
	poptPrintUsage(popt_context, stderr, 0);
	if (error)
		fprintf(stderr, "%s: %s\n", more, error);
	exit(exitcode);
}

int main(int argc, char **argv)
{
	static struct poptOption popt_options[] = {
		POPT_AUTOHELP
		{ "force", 'f', POPT_ARG_NONE, &opt_force, 0, N_("Force conversion (use with care)"), NULL },
		{ "resume", 'r', POPT_ARG_NONE, &opt_resume, 0, N_("Resume interrupted process"), NULL },
		POPT_TABLEEND
	};
	poptContext	popt_context;
	struct context ctx;
	const char *source = NULL;
	const char *dest = NULL;
	const char *arg;
	int in_progress;
	int r;

	setlocale(LC_ALL, "");
	bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
	textdomain(GETTEXT_PACKAGE);

	popt_context = poptGetContext(PACKAGE, argc, (const char **)argv,
	                              popt_options, 0);
	poptSetOtherOptionHelp(popt_context, N_("[OPTION...] <device> [<destination>]"));

	while((r = poptGetNextOpt(popt_context)) > 0) {
		/* optionally process arguments */
	}
	if (r < -1)
		usage(popt_context, 1, poptStrerror(r),
		      poptBadOption(popt_context, POPT_BADOPTION_NOALIAS));

	if (!(arg = poptGetArg(popt_context)))
		usage(popt_context, 1, _("Argument <device> missing."),
		      poptBadOption(popt_context, POPT_ERROR_NOARG));

	/*
	 * from here we could have sensible data in memory
	 * so protect it from being swapped out
	 */
	r = mlockall(MCL_CURRENT | MCL_FUTURE);
	if (r < 0) {
		perror(_("mlockall failed"));
		fprintf(stderr, _("WARNING!!! Possibly insecure memory. Are you root?\n"));
	}

	r = inspect_device(arg, &ctx);

	switch(r) {
		case -ENODEV:
			fprintf(stderr, _("Problem with device-mapper.\n"));
			return 1;
		case -EINVAL:
			fprintf(stderr, _("Found interrupted dmconvert device in an impossible state.\n"));
			return 1;
		case -EPERM:
			fprintf(stderr, _("Found active dmconvert state. Not touching.\n"));
			return 1;
		case -EBUSY:
			fprintf(stderr, _("Conversion already in progress.\n"));
			return 1;
		case 0:
			if (opt_resume) {
				fprintf(stderr, _("There is no device to resume.\n"));
				return 1;
			}

			source = strdup(arg);

			if (!(arg = poptGetArg(popt_context)))
				usage(popt_context, 1, _("Argument <destination> missing."),
				      poptBadOption(popt_context, POPT_ERROR_NOARG));

			dest = strdup(arg);

			ctx.size = 0;
			ctx.offset = 0;
			in_progress = 0;
			break;

		case 2:
			if (!opt_force) {
				fprintf(stderr, _("Conversion has been interrupted hard. Use force flag to force conversion.\nExpect data loss.\n"));
				return 1;
			}

		case 1:
			if (!opt_resume) {
				fprintf(stderr, _("Device is in a conversion state, but resume not flag given.\n"));
				return 1;
			}

			source = strdup(arg);

			arg = poptGetArg(popt_context);
			if (arg)
				fprintf(stderr, _("Warning: Argument <destination> ignored.\n"));

			dest = NULL;

			in_progress = 1;
			break;
	}

	poptFreeContext(popt_context);

	if (in_progress) {
		r = inspect_device(DM_DEV_NAME(ctx.temps[DMC_DEV_SOURCE]), NULL);
		switch(r) {
			case -ENODEV:
				fprintf(stderr, _("Problem with device-mapper.\n"));
				return 1;
			case -EEXIST:
				fprintf(stderr, _("Source device can't be a device that's currently being converted.\n"));
				return 1;
		}
	}

	r = inspect_device(in_progress ? DM_DEV_NAME(ctx.temps[DMC_DEV_DEST])
	                               : dest, NULL);
	switch(r) {
		case -ENODEV:
			fprintf(stderr, _("Problem with device-mapper.\n"));
			return 1;
		case -EEXIST:
			fprintf(stderr, _("Source device can't be a device that's currently being converted.\n"));
			return 1;
	}

	if (in_progress) {
		if (conversion_setup(&ctx, DM_DEV_NAME(ctx.temps[DMC_DEV_SOURCE]),
		                     DM_DEV_NAME(ctx.temps[DMC_DEV_DEST])) < 0)
			return 1;
	} else {
		if (conversion_setup(&ctx, source, dest) < 0)
			return 1;
	}

	if (conversion_init(&ctx, source, dest, in_progress) < 0)
		return 1;

	if (!in_progress && conversion_reopen_source(&ctx))
		return 1;

	free((void *)source);
	free((void *)dest);

	do {
		printf("\r%2d%% - %" PRIu64 "MB", (int)((ctx.offset * 100) / ctx.size), ctx.offset >> 20);
		fflush(stdout);

		r = conversion_step(&ctx);
	} while(r == 0);
	printf("\n");

	if (conversion_exit(&ctx) < 0)
		return 1;

	if (r < 0)
		return 1;

	return 0;
}
