#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <errno.h>
#include <libdevmapper.h>

#include "dmconvert.h"

int inspect_device(const char *dev, struct context *ctx)
{
	struct dm_task *dmt;
	void *next = NULL;
	uint64_t start, length, offset;
	unsigned int major, minor;
	pid_t pid, pid2;
	char buf[DM_NAME_MAX];
	char *target_type;
	char *params;
	char *p;
	int type;
	int in_progress = 0;
	int done = 0;
	int i;

	if (!(dmt = dm_task_create(DM_DEVICE_TABLE)))
		return -ENODEV;
	if (!dm_task_set_name(dmt, dev))
		return -ENODEV;
	if (!dm_task_run(dmt))
		return -ENODEV;

	next = dm_get_next_target(dmt, next, &start, &length,
	                          &target_type, &params);
	if (!target_type)
		return -ENODEV;
	if (strcmp(target_type, "linear") != 0)
		return 0;
	p = strchr(params, ' ');
	if (!p)
		return -ENODEV;
	*p++ = 0;
	if (sscanf(p, "%" PRIu64, &offset) != 1)
		return -ENODEV;

	if (parse_dm_dev(params, &major, &minor) < 0)
		return -ENODEV;
	if (!is_dm_major(major))
		return 0;
	type = lookup_dm_dev(buf, major, minor);
	if (type < 0)
		return (type == -ENOENT) ? 0 : type;
	if ((type = is_temp_dev(buf, &pid)) < 0)
		return 0;

	/*
	 * so, now we're sure this must be a
	 * device that was created by dmcrypt
	 */

	if (!ctx)
		return -EEXIST;

	for(i = 0; i < DMC_DEV_COUNT; i++)
		memset(&ctx->temps[i], 0, sizeof(struct dm_dev));

	if (start != 0 || offset != 0)
		return -EINVAL;
	switch(type) {
		case DMC_DEV_DEST:
			ctx->temps[DMC_DEV_DEST].name = strdup(buf);
			ctx->offset = length;
			break;
		case DMC_DEV_BOUNCE:
			ctx->temps[DMC_DEV_BOUNCE].name = strdup(buf);
			in_progress = 1;
			ctx->offset = 0;
			break;
		case DMC_DEV_SOURCE:
			return -EINVAL;
	}

	ctx->size = start + length;

	if (next)
		next = dm_get_next_target(dmt, next, &start, &length,
		                          &target_type, &params);
	else
		target_type = NULL;
	if (!target_type) {
		if (in_progress)
			goto check;
		return -EINVAL;
	} else if (done)
		return -EINVAL;

	if (strcmp(target_type, "linear") != 0)
		return -EINVAL;
	p = strchr(params, ' ');
	if (!p)
		return -ENODEV;
	*p++ = 0;
	if (sscanf(p, "%" PRIu64, &offset) != 1)
		return -ENODEV;

	if (parse_dm_dev(params, &major, &minor) < 0)
		return -ENODEV;
	type = lookup_dm_dev(buf, major, minor);
	if (type < 0)
		return -EINVAL;
	if ((type = is_temp_dev(buf, &pid2)) < 0)
		return -EINVAL;
	if (pid != pid2)
		return -EINVAL;

	switch(type) {
		case DMC_DEV_DEST:
			return -EINVAL;
		case DMC_DEV_BOUNCE:
			ctx->temps[DMC_DEV_BOUNCE].name = strdup(buf);
			if (in_progress)
				return -EINVAL;
			if (offset != 0 || start < ctx->size)
				return -EINVAL;
			in_progress = 1;
			break;
		case DMC_DEV_SOURCE:
			if (start != offset || start < ctx->size)
				return -EINVAL;
			ctx->temps[DMC_DEV_SOURCE].name = strdup(buf);
			done = 1;
			break;
	}

	ctx->size = start + length;

	if (next)
		next = dm_get_next_target(dmt, next, &start, &length,
		                          &target_type, &params);
	else
		target_type = NULL;
	if (!target_type) {
		if (done || in_progress)
			goto check;
		else
			return -EINVAL;
	}
	if (done || !in_progress)
		return -EINVAL;

	if (strcmp(target_type, "linear") != 0)
		return -EINVAL;
	p = strchr(params, ' ');
	if (!p)
		return -ENODEV;
	*p++ = 0;
	if (sscanf(p, "%" PRIu64, &offset) != 1)
		return -ENODEV;

	if (parse_dm_dev(params, &major, &minor) < 0)
		return -ENODEV;
	type = lookup_dm_dev(buf, major, minor);
	if (type < 0)
		return -EINVAL;
	if ((type = is_temp_dev(buf, &pid2)) < 0)
		return -EINVAL;
	if (pid != pid2)
		return -EINVAL;

	switch(type) {
		case DMC_DEV_DEST:
			return -EINVAL;
		case DMC_DEV_BOUNCE:
			return -EINVAL;
		case DMC_DEV_SOURCE:
			if (start != offset || start <= ctx->size)
				return -EINVAL;
			ctx->temps[DMC_DEV_SOURCE].name = strdup(buf);
			break;
	}

	ctx->size = start + length;

	if (next)
		dm_get_next_target(dmt, next, &start, &length,
		                   &target_type, &params);
	else
		target_type = NULL;

	if (target_type)
		return -EINVAL;

check:
	if (in_progress && kill(pid, 0) == 0)
		return -EBUSY;

	ctx->offset <<= SECTOR_SHIFT;
	ctx->size <<= SECTOR_SHIFT;

	return in_progress ? 2 : 1;
}

int setup_bounce_target(struct context *ctx)
{
	struct dm_task *dmt;
	struct dm_info info;

	if (!(dmt = dm_task_create(DM_DEVICE_CREATE)))
		return -ENODEV;
	if (!dm_task_set_name(dmt, DM_DEV_NAME(ctx->temps[DMC_DEV_BOUNCE])))
		goto out;
	if (!dm_task_run(dmt))
		goto out;
	if (!dm_task_get_info(dmt, &info))
		goto out;
	dm_task_destroy(dmt);
	ctx->temps[DMC_DEV_BOUNCE].major = info.major;
	ctx->temps[DMC_DEV_BOUNCE].minor = info.minor;

	return 0;

out:
	dm_task_destroy(dmt);
	return -ENODEV;
}

int capture_temp_targets(struct context *ctx, const struct dm_dev *temps)
{
	struct dm_task *dmt;
	struct dm_info info;

	if (!(dmt = dm_task_create(DM_DEVICE_RENAME)))
		return -ENODEV;
	if (!dm_task_set_name(dmt, DM_DEV_NAME(ctx->temps[DMC_DEV_SOURCE])))
		goto out;
	if (!dm_task_set_newname(dmt, DM_DEV_NAME(temps[DMC_DEV_SOURCE])))
		goto out;
	if (!dm_task_run(dmt))
		goto out;
	dm_task_destroy(dmt);
	free(ctx->temps[DMC_DEV_SOURCE].name);
	ctx->temps[DMC_DEV_SOURCE].name = strdup(temps[DMC_DEV_SOURCE].name);

	if (!(dmt = dm_task_create(DM_DEVICE_INFO)))
      return -ENODEV;
	if (!dm_task_set_name(dmt, DM_DEV_NAME(ctx->temps[DMC_DEV_SOURCE])))
      goto out;
	if (!dm_task_run(dmt))
		goto out;
	if (!dm_task_get_info(dmt, &info))
		goto out;
	dm_task_destroy(dmt);
	ctx->temps[DMC_DEV_SOURCE].major = info.major;
	ctx->temps[DMC_DEV_SOURCE].minor = info.minor;

	if (!(dmt = dm_task_create(DM_DEVICE_RENAME)))
		return -ENODEV;
	if (!dm_task_set_name(dmt, DM_DEV_NAME(ctx->temps[DMC_DEV_DEST])))
		goto out;
	if (!dm_task_set_newname(dmt, DM_DEV_NAME(temps[DMC_DEV_DEST])))
		goto out;
	if (!dm_task_run(dmt))
		goto out;
	dm_task_destroy(dmt);
	free(ctx->temps[DMC_DEV_DEST].name);
	ctx->temps[DMC_DEV_DEST].name = strdup(temps[DMC_DEV_DEST].name);

	if (!(dmt = dm_task_create(DM_DEVICE_INFO)))
      return -ENODEV;
	if (!dm_task_set_name(dmt, DM_DEV_NAME(ctx->temps[DMC_DEV_DEST])))
      goto out;
	if (!dm_task_run(dmt))
		goto out;
	if (!dm_task_get_info(dmt, &info))
		goto out;
	dm_task_destroy(dmt);
	ctx->temps[DMC_DEV_DEST].major = info.major;
	ctx->temps[DMC_DEV_DEST].minor = info.minor;

	if (DM_DEV_NAME(ctx->temps[DMC_DEV_BOUNCE])) {
		if (!(dmt = dm_task_create(DM_DEVICE_RENAME)))
			return -ENODEV;
		if (!dm_task_set_name(dmt, DM_DEV_NAME(ctx->temps[DMC_DEV_BOUNCE])))
			goto out;
		if (!dm_task_set_newname(dmt, DM_DEV_NAME(temps[DMC_DEV_BOUNCE])))
			goto out;
		if (!dm_task_run(dmt))
			goto out;
		dm_task_destroy(dmt);
		free(ctx->temps[DMC_DEV_BOUNCE].name);
		ctx->temps[DMC_DEV_BOUNCE].name = strdup(temps[DMC_DEV_BOUNCE].name);

		if (!(dmt = dm_task_create(DM_DEVICE_INFO)))
	      return -ENODEV;
		if (!dm_task_set_name(dmt, DM_DEV_NAME(ctx->temps[DMC_DEV_BOUNCE])))
	      goto out;
		if (!dm_task_run(dmt))
			goto out;
		if (!dm_task_get_info(dmt, &info))
			goto out;
		dm_task_destroy(dmt);
		ctx->temps[DMC_DEV_BOUNCE].major = info.major;
		ctx->temps[DMC_DEV_BOUNCE].minor = info.minor;
	} else {
		ctx->temps[DMC_DEV_BOUNCE].name = strdup(temps[DMC_DEV_BOUNCE].name);
		if (setup_bounce_target(ctx) < 0)
			return -ENODEV;
	}

	return 0;

out:
	dm_task_destroy(dmt);
	return -ENODEV;
}

int swap_source_table(struct context *ctx, int dir)
{
	struct dm_task *dmt1 = NULL;
	struct dm_task *dmt2 = NULL;
	struct dm_info info;
	char tmp[DM_DEV_BUFSIZE];
	uint64_t start, length;
	char *target_type;
	char *params;
	void *next = NULL;

	if (!dir) {
		if (!(dmt1 = dm_task_create(DM_DEVICE_CREATE)))
			goto out;
		if (!dm_task_set_name(dmt1, DM_DEV_NAME(ctx->temps[DMC_DEV_SOURCE])))
			goto out;
		if (!dm_task_run(dmt1))
			goto out;
		if (!dm_task_get_info(dmt1, &info))
			goto out;
		dm_task_destroy(dmt1);
		ctx->temps[DMC_DEV_SOURCE].major = info.major;
		ctx->temps[DMC_DEV_SOURCE].minor = info.minor;
	}

	if (!(dmt1 = dm_task_create(DM_DEVICE_TABLE)))
		goto out;
	if (!dm_task_set_name(dmt1, dir ? DM_DEV_NAME(ctx->temps[DMC_DEV_DEST])
	                                : ctx->source))
		goto out;
	if (!dm_task_run(dmt1))
		goto out;

	if (!(dmt2 = dm_task_create(DM_DEVICE_RELOAD)))
		goto out;
	if (!dm_task_set_name(dmt2, dir ? ctx->source
	                                : DM_DEV_NAME(ctx->temps[DMC_DEV_SOURCE])))
		goto out;

	do {
		next = dm_get_next_target(dmt1, next, &start, &length,
		                          &target_type, &params);
		if (!target_type)
			break;
		if (!dm_task_add_target(dmt2, start, length, target_type, params))
			goto out;
	} while(next);

	if (!dm_task_run(dmt2))
		goto out;

	dm_task_destroy(dmt1);
	dm_task_destroy(dmt2);

	if (!(dmt1 = dm_task_create(DM_DEVICE_RESUME)))
		goto out;
	if (!dm_task_set_name(dmt1, dir ? ctx->source
	                                : DM_DEV_NAME(ctx->temps[DMC_DEV_SOURCE])))
		goto out;
	if (!dm_task_run(dmt1))
		goto out;
	dm_task_destroy(dmt1);

	return 0;

out:
	if (dmt1)
		dm_task_destroy(dmt1);
	if (dmt2)
		dm_task_destroy(dmt2);
	return -ENODEV;
}

int remove_temp_targets(struct context *ctx)
{
	struct dm_task *dmt;
	int r = 0;
	int i;

	for(i = 0; i < DMC_DEV_COUNT; i++) {
		if (!(dmt = dm_task_create(DM_DEVICE_REMOVE)))
			goto bad;
		if (!dm_task_set_name(dmt, DM_DEV_NAME(ctx->temps[i])))
			goto bad;
		if (!dm_task_run(dmt))
			goto bad;

		dm_task_destroy(dmt);
		continue;
	bad:
		r = -ENODEV;
		dm_task_destroy(dmt);
	}

	return r;
}

int remove_bounce_target(struct context *ctx)
{
	struct dm_task *dmt;

	if (!(dmt = dm_task_create(DM_DEVICE_REMOVE)))
		return -ENODEV;
	if (!dm_task_set_name(dmt, DM_DEV_NAME(ctx->temps[DMC_DEV_BOUNCE])))
		goto out;
	if (!dm_task_run(dmt))
		goto out;
	dm_task_destroy(dmt);

	return 0;

out:
	dm_task_destroy(dmt);
	return -ENODEV;
}

int rename_dest_target(struct context *ctx)
{
	struct dm_task *dmt;
	struct dm_info info;

	if (!(dmt = dm_task_create(DM_DEVICE_RENAME)))
		return -ENODEV;
	if (!dm_task_set_name(dmt, ctx->dest))
		goto out;
	if (!dm_task_set_newname(dmt, DM_DEV_NAME(ctx->temps[DMC_DEV_DEST])))
		goto out;
	if (!dm_task_run(dmt))
		goto out;
	dm_task_destroy(dmt);

	free(ctx->dest);
	ctx->dest = NULL;

	if (!(dmt = dm_task_create(DM_DEVICE_INFO)))
      return -ENODEV;
	if (!dm_task_set_name(dmt, DM_DEV_NAME(ctx->temps[DMC_DEV_DEST])))
      goto out;
	if (!dm_task_run(dmt))
		goto out;
	if (!dm_task_get_info(dmt, &info))
		goto out;
	dm_task_destroy(dmt);
	ctx->temps[DMC_DEV_DEST].major = info.major;
	ctx->temps[DMC_DEV_DEST].minor = info.minor;

	return 0;

out:
	dm_task_destroy(dmt);
	return -ENODEV;
}

int start_bounce_target(struct context *ctx)
{
	const char *dmdir = dm_dir();
	struct dm_task *dmt;
	char buf[DM_NAME_MAX + strlen(dmdir) + 32];
	char tmp[DM_DEV_BUFSIZE];
	uint64_t size = ctx->size - ctx->offset;
	if (size > ((uint64_t)ctx->bsize * ctx->blocks))
		size = (uint64_t)ctx->bsize * ctx->blocks;

	if (!(dmt = dm_task_create(DM_DEVICE_RELOAD)))
		return -ENODEV;
	if (!dm_task_set_name(dmt, DM_DEV_NAME(ctx->temps[DMC_DEV_BOUNCE])))
		goto out;
	sprintf(buf, "%s %" PRIu64,
	        DM_DEV_FORMAT(tmp, ctx->temps[DMC_DEV_SOURCE]),
	        ctx->offset >> SECTOR_SHIFT);
	if (!dm_task_add_target(dmt, 0, size >> SECTOR_SHIFT,
	                        "linear", buf))
		goto out;
	if (!dm_task_run(dmt))
		goto out;
	dm_task_destroy(dmt);

	if (!(dmt = dm_task_create(DM_DEVICE_RESUME)))
		return -ENODEV;
	if (!dm_task_set_name(dmt, DM_DEV_NAME(ctx->temps[DMC_DEV_BOUNCE])))
		goto out;
	if (!dm_task_run(dmt))
		goto out;
	dm_task_destroy(dmt);

	return 0;

out:
	dm_task_destroy(dmt);
	return -ENODEV;
}

int step_bounce_target(struct context *ctx)
{
	struct dm_task *dmt;

	if (!(dmt = dm_task_create(DM_DEVICE_SUSPEND)))
		return -ENODEV;
	if (!dm_task_set_name(dmt, DM_DEV_NAME(ctx->temps[DMC_DEV_BOUNCE])))
		goto out;
	if (!dm_task_run(dmt))
		goto out;
	dm_task_destroy(dmt);

	return 0;

out:
	dm_task_destroy(dmt);
	return -ENODEV;
}

int flush_bounce_target(struct context *ctx, uint64_t done)
{
	const char *dmdir = dm_dir();
	struct dm_task *dmt;
	char buf[DM_NAME_MAX + strlen(dmdir) + 32];
	char tmp[DM_DEV_BUFSIZE];
	uint64_t size = ctx->size - (ctx->offset - done);
	uint64_t remaining;
	if (size > ((uint64_t)ctx->bsize * ctx->blocks))
		size = (uint64_t)ctx->bsize * ctx->blocks;
	remaining = size - done;

	if (!(dmt = dm_task_create(DM_DEVICE_RELOAD)))
		return -ENODEV;
	if (!dm_task_set_name(dmt, DM_DEV_NAME(ctx->temps[DMC_DEV_BOUNCE])))
		goto out;

	if (done) {
		sprintf(buf, "%s %" PRIu64,
		        DM_DEV_FORMAT(tmp, ctx->temps[DMC_DEV_DEST]),
		        (ctx->offset - done) >> SECTOR_SHIFT);
		if (!dm_task_add_target(dmt, 0, done >> SECTOR_SHIFT,
		                        "linear", buf))
			goto out;
	}
	if (remaining) {
		sprintf(buf, "%s %" PRIu64,
		        DM_DEV_FORMAT(tmp, ctx->temps[DMC_DEV_SOURCE]),
		        ctx->offset >> SECTOR_SHIFT);
		if (!dm_task_add_target(dmt, done >> SECTOR_SHIFT,
		                        remaining >> SECTOR_SHIFT, "linear", buf))
			goto out;
	}

	if (!dm_task_run(dmt))
		goto out;
	dm_task_destroy(dmt);

	if (!(dmt = dm_task_create(DM_DEVICE_RESUME)))
		return -ENODEV;
	if (!dm_task_set_name(dmt, DM_DEV_NAME(ctx->temps[DMC_DEV_BOUNCE])))
		goto out;
	if (!dm_task_run(dmt))
		goto out;
	dm_task_destroy(dmt);

	return 0;

out:
	dm_task_destroy(dmt);
	return -ENODEV;
}

static int do_device_bounce(struct context *ctx, uint64_t done,
                            uint64_t bounce, uint64_t remaining)
{
	const char *dmdir = dm_dir();
	struct dm_task *dmt;
	char buf[DM_NAME_MAX + strlen(dmdir) + 32];
	char tmp[DM_DEV_BUFSIZE];

	if (!(dmt = dm_task_create(DM_DEVICE_RELOAD)))
		return -ENODEV;
	if (!dm_task_set_name(dmt, ctx->source))
		goto out;

	if (done) {
		sprintf(buf, "%s %" PRIu64,
		        DM_DEV_FORMAT(tmp, ctx->temps[DMC_DEV_DEST]), (uint64_t)0);
		if (!dm_task_add_target(dmt, 0, done  >> SECTOR_SHIFT,
		                        "linear", buf))
			goto out;
	}
	if (bounce) {
		sprintf(buf, "%s %" PRIu64,
		        DM_DEV_FORMAT(tmp, ctx->temps[DMC_DEV_BOUNCE]), (uint64_t)0);
		if (!dm_task_add_target(dmt, done >> SECTOR_SHIFT,
		                        bounce >> SECTOR_SHIFT, "linear", buf))
			goto out;
	}
	if (remaining) {
		sprintf(buf, "%s %" PRIu64,
		        DM_DEV_FORMAT(tmp, ctx->temps[DMC_DEV_SOURCE]),
		        (done + bounce) >> SECTOR_SHIFT);
		if (!dm_task_add_target(dmt, (done + bounce) >> SECTOR_SHIFT,
		                        remaining >> SECTOR_SHIFT, "linear", buf))
			goto out;
	}

	if (!dm_task_run(dmt))
		goto out;
	dm_task_destroy(dmt);

	if (!(dmt = dm_task_create(DM_DEVICE_RESUME)))
		return -ENODEV;
	if (!dm_task_set_name(dmt, ctx->source))
		goto out;
	if (!dm_task_run(dmt))
		goto out;
	dm_task_destroy(dmt);

	return 0;

out:
	dm_task_destroy(dmt);
	return -ENODEV;
}

int init_device_bounce(struct context *ctx)
{
	uint64_t size = ctx->size - ctx->offset;
	if (size > ((uint64_t)ctx->bsize * ctx->blocks))
		size = (uint64_t)ctx->bsize * ctx->blocks;

	return do_device_bounce(ctx, ctx->offset, size,
	                        ctx->size - ctx->offset - size);
}

int finish_device_bounce(struct context *ctx)
{
	return do_device_bounce(ctx, ctx->offset, 0, ctx->size - ctx->offset);
}
