#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>

#include <libdevmapper.h>

#include "dmconvert.h"

#undef DEV_PARSE_HACK

#define MAX_MAJORS 16

static unsigned int _majors[MAX_MAJORS];
static unsigned int _majorcnt = 0;

static char *_dmdev_names[] = {
	DMC_NAME_BASE "-" DMC_NAME_SOURCE "-",
	DMC_NAME_BASE "-" DMC_NAME_DEST   "-",
	DMC_NAME_BASE "-" DMC_NAME_BOUNCE "-",
	NULL
};

/*
 * Return the major numbers that were allocated to device-mapper.
 */
static int get_dm_majors(unsigned int *majors, int max) {
	char line[128];
	char major_name[32];
	unsigned int major;
	int i = 0;
	FILE *file = fopen("/proc/devices", "r");

	if (!file)
		return -ENOENT;

	do {
		if (!fgets(line, sizeof(line) - 1, file)) {
			fclose(file);
			return -ENOENT;
		}
		line[sizeof(line) - 1] = 0;
	} while(strncmp(line, "Block", 5) != 0);

	while(fgets(line, sizeof(line) - 1, file)) {
		if (sscanf(line, "%u %31s\n", &major, major_name) != 2)
			continue;
		if (strcmp(major_name, DM_KERNEL_NAME) != 0)
			continue;
		majors[i] = major;
		if (++i >= max)
			break;
	}

	fclose(file);

	return i;
}

int is_dm_major(unsigned int major) {
	int i;

	if (!_majorcnt) {
		int r = get_dm_majors(_majors, MAX_MAJORS);
		if (r <= 0) {
			fprintf(stderr, "Could not get device-mapper major from /proc/devices\n");
			exit(1);
		}
		_majorcnt = r;
	}

	for(i = 0; i < _majorcnt; i++)
		if (_majors[i] == major)
			return 1;

	return 0;
}

unsigned int first_dm_major(void) {
	if (!_majorcnt) {
		int r = get_dm_majors(_majors, MAX_MAJORS);
		if (r <= 0) {
			fprintf(stderr, "Could not get device-mapper major from /proc/devices\n");
			exit(1);
		}
		_majorcnt = r;
	}

	return _majors[0];
}

/*
 * Parse the device string returned by
 * libdevmapper into major and minor.
 */
int parse_dm_dev(const char *dev, unsigned int *major, unsigned int *minor)
{
#ifdef DEV_PARSE_HACK
	if (isxdigit(dev[0]) &&
	    isxdigit(dev[1]) &&
	    dev[2] == ':' &&
	    isxdigit(dev[3]) &&
	    isxdigit(dev[4])) { /* 2.4 format */
		if (sscanf(dev, "%02x:%02x", major, minor) != 2)
			return -EINVAL;
	} else if (isdigit(dev[0]) &&
	           isdigit(dev[1]) &&
	           isdigit(dev[2]) &&
	           strchr(dev, ':')) { /* >= 2.6.0-test3 format */
#endif
		if (sscanf(dev, "%u:%u", major, minor) != 2)
			return -EINVAL;
#ifdef DEV_PARSE_HACK
	} else { /* >= 2.5.69 format */
		struct stat st;
		char buf[256];

		snprintf(buf, sizeof(buf) - 1, "/dev/%s", dev);
		buf[sizeof(buf) - 1] = '\0';

		if (stat(buf, &st) < 0) {
			/* now it's getting really ugly */
			if (sscanf(dev, "dm-%u", minor) == 1)
				*major = first_dm_major();
			else
				return -ENOENT;
		} else {
			if (!S_ISBLK(st.st_mode))
				return -ENOTBLK;
			*major = major(st.st_rdev);
			*minor = minor(st.st_rdev);
		}
	}
#endif

	return 0;
}

int lookup_dm_dev(char *dest, unsigned int major, unsigned int minor)
{
	struct dm_task *dmt;
	struct dm_names *names;
	unsigned int next = 0;
	int found = 0;

	if (!(dmt = dm_task_create(DM_DEVICE_LIST)))
		return -EINVAL;
	if (!dm_task_run(dmt))
		goto out;
	if (!(names = dm_task_get_names(dmt)))
		goto out;

	if (names->dev)
		do {
			names = (void *)names + next;
			if (major(names->dev) == major &&
			    minor(names->dev) == minor) {
				found = 1;
				strncpy(dest, names->name, DM_NAME_MAX - 1);
				dest[DM_NAME_MAX - 1] = '\0';
				break;
			}
			next = names->next;
		} while(next);

	dm_task_destroy(dmt);
	return found ? 0 : -ENOENT;

out:
	dm_task_destroy(dmt);
	return -EINVAL;
}

int is_temp_dev(const char *dev, pid_t *pid)
{
	char **name;
	pid_t _pid;

	for(name = _dmdev_names; *name; name++)
		if (strncmp(dev, *name, strlen(*name)) == 0)
			break;

	if (!*name)
		return -ENOENT;

	if (sscanf(dev + strlen(*name), "%u", &_pid) != 1)
		return -EINVAL;

	if (pid)
		*pid = _pid;

	return name - _dmdev_names;
}

void setup_temp_names(struct dm_dev *list) {
	char buf[DM_NAME_MAX];
	pid_t pid = getpid();
	char **name;

	for(name = _dmdev_names; *name; name++) {
		snprintf(buf, DM_NAME_MAX - 1, "%s%d", *name, (int)pid);
		buf[DM_NAME_MAX - 1] = '\0';
		(list++)->name = strdup(buf);
	}
}
